# -*- coding: utf-8 -*-
"""Untitled27.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1B9SZoZrU-cBPcNbOTgAVuCVQKlpso2-R
"""

"""
Run Experiment 1: Comparison vs Optimal for Varying Rho_G
---------------------------------------------------------
This script executes Experiment 1 from the matrix product approximation study.
It generates matrices with different Rho_G characteristics, runs various
approximation algorithms and computes theoretical bounds, then plots the results.
"""

import numpy as np
import pandas as pd
import os
import warnings

# Import necessary functions from the core library
from matrix_product_approximations_exp1 import (
    generate_matrices_for_rho,
    run_experiment_iterations,
    plot_experiment_results,
    PLOT_STYLES,
    ExperimentFailureError
)

def main_experiment1(n_dim=20,m_dim=50,p_dim=30,target_rho_values = [1.0, 5.0, 20.0]):
    """
    Main function to configure and run Experiment 1.
    """
    print("--- Running Experiment 1: Comparison vs Optimal for Varying Rho_G ---")

    # Suppress specific warnings for cleaner output during experiments
    warnings.filterwarnings("ignore", category=RuntimeWarning, module="cvxpy") # CVXPY specific warnings
    warnings.filterwarnings("ignore", category=UserWarning, message="Solution may be inaccurate.*") # CVXPY SCS solver
    warnings.filterwarnings("ignore", category=RuntimeWarning, message="invalid value encountered in sqrt") # From np.sqrt(negative)
    warnings.filterwarnings("ignore", category=RuntimeWarning, message="divide by zero encountered in true_divide") # Division by zero
    warnings.filterwarnings("ignore", category=RuntimeWarning, message="invalid value encountered in true_divide")

    # k_values: Sparsity levels / sketch sizes to test
    # Example: 7 values from 10% to 40% of n_dim
    k_values_exp1 = np.round(np.linspace(0.1 * n_dim, 0.4 * n_dim, 7)).astype(int)
    k_values_exp1 = np.unique(np.concatenate(([0], k_values_exp1[k_values_exp1 > 0]))).tolist() # Add k=0, ensure unique and sorted

    num_trials_randomized_algos = 50  # Number of trials for randomized algorithms (e.g., 100+ for publication)

    # Optimal v_k* computation settings
    compute_optimal_vk_star = True # Set to False to speed up if n is large or C(n,k) is too high
    # Max C(n,k) combinations for v_k* before skipping (adjust based on patience/compute power)
    vk_star_combination_threshold = 1000000 # Default was 100,000,000, reduced for faster example

    # --- End of Editable Parameters ---

    # Dictionary to store results for each experimental setup (each target Rho_G)
    all_experiment_results = {}

    # Loop through each target Rho_G value
    for target_rho in target_rho_values:
        print(f"\n--- Processing Target Rho_G ≈ {target_rho:.1f} ---")

        # Generate matrices A and B aiming for the target_rho
        # base_seed is varied per target_rho for diversity in matrix generation
        generated_matrix_data = generate_matrices_for_rho(
            target_rho=target_rho,
            m=m_dim,
            p=p_dim,
            n=n_dim,
            tolerance=0.35,  # Allow some deviation from the target Rho_G
            max_attempts=3000,
            base_seed=42 + int(target_rho * 10) # Vary seed based on rho
        )

        if generated_matrix_data is None:
            print(f"Failed to generate suitable matrices for target Rho_G={target_rho}. Skipping this value.")
            continue

        A_exp, B_exp, actual_rho_achieved = generated_matrix_data
        experiment_name = f"Rho_{actual_rho_achieved:.1f}" # Name for storing/plotting

        print(f"Running experiment for: {experiment_name} (Target Rho_G={target_rho:.1f}, Actual Rho_G={actual_rho_achieved:.2f})")
        print(f"Matrix dimensions: A({A_exp.shape}), B({B_exp.shape})")
        print(f"k values: {k_values_exp1}")

        try:
            # Run the core experiment iterations for the generated A and B
            current_results = run_experiment_iterations(
                A_matrix=A_exp,
                B_matrix=B_exp, # B_exp is p x n
                k_values_list=k_values_exp1,
                num_trials=num_trials_randomized_algos,
                compute_vk_star_flag=compute_optimal_vk_star,
                vk_star_comb_threshold=vk_star_combination_threshold,
                run_algorithms_flag=True,
                run_bounds_flag=True
            )

            all_experiment_results[experiment_name] = current_results

            # Save results to a CSV file for this specific experiment run
            # Ensure 'results' directory exists (created by matrix_product_approximations.py)
            results_df = pd.DataFrame({
                'k': current_results['k'],
                'k_ratio': current_results['k'] / n_dim if n_dim > 0 else 0,
                'Optimal_Error_v_k_star': current_results['Optimal Error v_k^*'],
                'Error_Leverage_Score': current_results['Error Leverage Score (Actual)'],
                'Error_CountSketch': current_results['Error CountSketch (Actual)'],
                'Error_SRHT': current_results['Error SRHT (Actual)'],
                'Error_Gaussian_Proj': current_results['Error Gaussian (Actual)'],
                'Error_Greedy_OMP': current_results['Error Greedy OMP (Actual)'],
                'Bound_QP_CVXPY_Best': current_results['Your Bound (QP CVXPY Best)'],
                'Bound_QP_Analytical': current_results['Your Bound (QP Analytical)'],
                'Bound_Binary': current_results['Your Bound (Binary)'],
                'Bound_Leverage_Score_Exp': current_results['Bound (Leverage Score Exp.)'],
                'Bound_Sketching_Simple': current_results['Bound (Sketching Simple)'],
                'Actual_Rho_G': actual_rho_achieved,
                'Frob_ABt_Sq': current_results['Frob_ABt_Sq']
            })

            csv_filename = f"experiment1_results_{experiment_name}.csv"
            results_df.to_csv(os.path.join("results", csv_filename), index=False)
            print(f"Saved results to: results/{csv_filename}")

        except ExperimentFailureError as e:
            print(f"Experiment failed critically for {experiment_name}: {e}")
            print("Skipping this experimental setup.")
            continue
        except Exception as e:
            print(f"An unexpected error occurred during experiment for {experiment_name}: {e}")
            import traceback
            traceback.print_exc()
            print("Skipping this experimental setup.")
            continue


    # After all Rho_G values are processed, plot the combined results
    if all_experiment_results:
        print("\n--- Plotting Combined Experiment 1 Results ---")
        plot_experiment_results(
            results_by_name=all_experiment_results,
            main_plot_filename_base="experiment1_comparison", # Filename for plot image
            plot_styles_dict=PLOT_STYLES,
            y_axis_label_flag=True, # Include Y-axis label on the first subplot
            figure_super_title="Experiment 1: Comparison vs Optimal for Varying " + r"$\rho_G$"
        )
        print("\nExperiment 1 completed and results plotted.")
    else:
        print("\nNo successful experiments were run. No plots will be generated.")

    print("\n=== All configured experiments finished. ===")


if __name__ == "__main__":
    # --- Experiment Parameters (Editable) ---
    n_dim = 10       # Number of columns (n) for matrices A and B
    m_dim = 50      # Number of rows (m) for matrix A
    p_dim = 15       # Number of rows (p) for matrix B (B is p x n)
    # Target Rho_G values to generate matrices for (e.g., low, medium, high coherence)
    target_rho_values = [1.0, 5.0, 20.0]
    main_experiment1(n_dim,m_dim,p_dim,target_rho_values)

